/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#include "rcm_class.h"
#include <QtSerialPort/QSerialPort>
#include <QNetworkInterface>
#include <iostream>
#include <QDebug>


//
// TODO:
// 1. Error Handler that connects to serialPort Error?
// 2. Monitoring RCM connection validity... I guess that would be part of the Error Handler?
// 3. Room for improvement on RX/TX indication in console.
//

/* Constructor */
RCM_Class::RCM_Class(QList<RF_Channel*> rf_channel_list) :
	RF_Channels{rf_channel_list}
{
	/* Get the names of the ports, and other startup values from config.txt */
	config = new ConfigHandler(QCoreApplication::applicationDirPath() + "/config.txt");

	/* Create instances of the various interfacing classes */
	RCM_port = new QSerialPort();
	tcpServer = new QTcpServer();
	tcpSocket = new QTcpSocket();

	/* Remark: Communication signals and slots are handled when a connection occurs, inside the handler. */
	connect(tcpServer, &QTcpServer::newConnection, this, &RCM_Class::RCM_TCP_newConnection_Handler);

	//
	// TODO:
	// ONLY WINDOWS HAS THE ERROR OCCURED SIGNAL!
	// LINUX Environment uses Qt 5.11.3, which does not yet have the errorOccurred signal implemented into the Qt Framework.
	// Ideally everything should be upgraded to 5.15 (LTS) if possible, but I don't know if it won't retroactively break existing images...
	//
	/* Connect The TCP Socket handler for new connections */
#if defined (Q_OS_WINDOWS)
	connect(tcpSocket, &QTcpSocket::errorOccurred, this, &RCM_Class::Handler_RCM_TCP_Socket_Error);
#endif

#if defined(Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835())
	{
		GPIO_HAT_serial_mode = new GPIOClass(QString::number(4));
		GPIO_HAT_serial_mode->setDirection("out");
	}
#endif

}

RCM_Class::~RCM_Class()
{
	RCM_Close();
	Disconnect_Signals_Slots();
}

/* Kill RCM quick and dirty */
void RCM_Class::RCM_kill()
{
	 RCM_Close();
	 Disconnect_Signals_Slots();
}

/**********************************************************************************************************************************************************************************
 * SETUP FUNCTIONS
 *********************************************************************************************************************************************************************************/
 /* Setup the serial connection settings */
void RCM_Class::Setup(QString portname)
{
	RCM_port->setPortName(portname);      //COM PORT
	RCM_port->setBaudRate(QSerialPort::Baud115200);
	RCM_port->setDataBits(QSerialPort::Data8);
	RCM_port->setParity(QSerialPort::NoParity);
	RCM_port->setStopBits(QSerialPort::OneStop);
	RCM_port->setFlowControl(QSerialPort::NoFlowControl);
}

bool RCM_Class::RCM_Open()
{
	if (RCM_port->isOpen())
	{
		RCM_port->close();
	}

	if (RCM_port->open(QIODevice::ReadWrite))
	{
		qDebug() << "RCM:" << RCM_port->portName() << "-> Connection established";
		return true;
	}

	return false;
}

void RCM_Class::RCM_Close()
{
	/* Close Serial Connections */
	if (RCM_port->isOpen())
	{
		RCM_port->close();
		qDebug() << "RCM:" << RCM_port->portName() << "-> Connection closed";
	}

	/* Close TCP server and stop listening */
	if (tcpServer != nullptr)
	{
		if (tcpServer->isListening())
		{
			tcpServer->close();
			emit signal_RCM_notification("TCP Server has stopped listening for connections.");
			qDebug() << "RCM: TCP Server has stopped listening for connections.";
		}
	}

	/* Disconnect existing connections and close the TCP socket */
	if (tcpSocket != nullptr)
	{
		if (tcpSocket->isOpen())
		{
			RCM_TCP_Kick_Client();
			tcpSocket->close();
			qDebug() << "RCM: TCP Socket connection closed.";
		}
	}
}


/* Disconnect all signals and slots related to communication */
void RCM_Class::Disconnect_Signals_Slots()
{
	/* Disconnect RCM signals and slots that are managed within RF_System */
	disconnect(RCM_port, &QSerialPort::readyRead, this, &RCM_Class::RCM_Live_ReadyRead_Handler);		//Serial Interfaces Live Handler
	disconnect(RCM_port, &QSerialPort::readyRead, this, &RCM_Class::RCM_Blind_ReadyRead_Handler);		//Serial Interfaces Blind Handler

	disconnect(tcpSocket, &QTcpSocket::readyRead, this, &RCM_Class::RCM_Live_ReadyRead_Handler);		//TCP Interfaces Live Handler
	disconnect(tcpSocket, &QTcpSocket::readyRead, this, &RCM_Class::RCM_Blind_ReadyRead_Handler);		//TCP Interface Blind Handler


	/* If Applicable - Switch the Serial interface selector GPIO to RS232 Mode */
#if defined (Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835())
	{
		GPIO_HAT_serial_mode->setValue("0");		//Sets RS232 mode
	}
#endif

	/* Disconnect RCM signals and slots to/from RF channels */
	for (int i = 0; i < RF_Channels.count(); i++)
	{
		/* Disconnect Live RCM signals and slots */
		disconnect(this, &RCM_Class::signal_RCM_message, RF_Channels.at(i), &RF_Channel::RCM_Live_serial_writeRead_to_SGx);
		disconnect(this, &RCM_Class::signal_RCM_sweep_message, RF_Channels.at(i), &RF_Channel::RCM_Live_serial_Sweep_handler);
		disconnect(RF_Channels.at(i), &RF_Channel::signal_RCM_serial_response, this, &RCM_Class::RCM_Live_Response_from_SGX_Handler);

		/* Disconnect Blind RCM signals and slots */
		disconnect(this, &RCM_Class::signal_RCM_message, RF_Channels.at(i), &RF_Channel::RCM_Blind_serial_write_to_SGx);
		disconnect(RF_Channels.at(i)->SGX_port->serial, &QSerialPort::readyRead, RF_Channels.at(i), &RF_Channel::RCM_Blind_serial_handleResponse_from_SGx);
		disconnect(RF_Channels.at(i), &RF_Channel::signal_RCM_serial_response, this, &RCM_Class::RCM_Blind_Response_from_SGX_Handler);
		disconnect(RF_Channels.at(i), &RF_Channel::signal_RCM_serial_response, this, &RCM_Class::RCM_Blind_Response_from_SGX_Handler);

		/* Reset RF Channel's error handler (and reset detection in case of Live RCM) */
		qDebug() << "RCM: Resetting RF Channel[" << i+1 << "]'s Serial_Error_Handler.";
		disconnect(RCM_Class::RF_Channels.at(i)->SGX_port->serial, &QSerialPort::errorOccurred, RCM_Class::RF_Channels.at(i), &RF_Channel::Serial_Error_Handler);
		connect(RCM_Class::RF_Channels.at(i)->SGX_port->serial, &QSerialPort::errorOccurred, RCM_Class::RF_Channels.at(i), &RF_Channel::Serial_Error_Handler);
	}

	disconnect(tcpSocket, &QTcpSocket::disconnected, this, &RCM_Class::RCM_TCP_Client_Disconnected);
}

/* Handler for communication errors */
void RCM_Class::Handler_RCM_Error()
{
	/* Communicate the Error */
	qDebug() << "RCM: Error" << RCM_port->error() << "-" << RCM_port->errorString();

	emit signal_RCM_error(RCM_port->error(), RCM_port->errorString(), RCM_port->portName());
	emit signal_RCM_notification("Error " + QString::number(RCM_port->error()) + " - " + RCM_port->errorString() + " (" + RCM_port->portName() + ")");

	/* Switch the state to RCM_OFF */
	Configure_Remote_Command_Mode(RCM_MODE::RCM_MODE_OFF);
}

void RCM_Class::Handler_RCM_TCP_Server_Error()
{
	qDebug() << "RCM: Server Error" << tcpServer->serverError() << "-" << tcpServer->errorString();
	emit signal_RCM_notification("TCP Server Error " + QString::number(tcpServer->serverError()) + " - " + tcpServer->errorString());

	/* Switch the state to RCM_OFF */
	Configure_Remote_Command_Mode(RCM_MODE::RCM_MODE_OFF);
}

void RCM_Class::Handler_RCM_TCP_Socket_Error()
{
	qDebug() << "RCM: Socket Error" << tcpSocket->error() << "-" << tcpSocket->errorString();
	emit signal_RCM_notification("TCP Socket Error " + QString::number(tcpSocket->error()) + " - " + tcpSocket->errorString());

	/* Switch the state to RCM_OFF */
	Configure_Remote_Command_Mode(RCM_MODE::RCM_MODE_OFF);
}


/* Handler for switching between different RCM modes
 * Connect a signal that selects a remote command mode from GUI / RF System to this slot function. */
void RCM_Class::Handler_Remote_Command_Mode_Set(RCM_MODE mode)
{
	/* Ignore mode select requests if the mode is already selected. */
	if (mode != last_RCM_mode)
	{
		Configure_Remote_Command_Mode(mode);
	}
//	else
//	{
//		qDebug() << "RCM: The selected remote command mode is already active";
//	}
}


void RCM_Class::Configure_Remote_Command_Mode(RCM_MODE mode)
{
	qDebug() << "RCM: Configuring RCM mode" << mode;

	/* Always need to disable the port and disconnect signals and slots first */
	RCM_Close();
	Disconnect_Signals_Slots();

	QString notification = "";
	QString port = "";

	switch(mode)
	{
		/* Close RCM port and disconnect all signals and slots. */
		case RCM_MODE::RCM_MODE_OFF:

			/* Restart normal RF Channel functionality */
			for (int i = 1; i <= RF_Channels.count(); i++)
			{
				emit signal_timerStart(i, config->get_polling_rate_data());
				emit signal_set_error_clearing_enable(i, true);
			}

			emit signal_RCM_notification("Remote command mode disabled.");
			break;


		case RCM_MODE::RCM_MODE_USB_LIVE:

			/* Disable Auto-Error Clearing, ensure RF Channel is doing its polling actions. */
			for (int i = 1; i <= RF_Channels.count(); i++)
			{
				emit signal_set_error_clearing_enable(i, false);
				emit signal_timerStart(i, config->get_polling_rate_data());		//Enable Readings
			}

			/* Wait for above signals to take effect */
			//
			// TODO:
			// This msleep action is inelegant and not 100% reliable.
			// There should be a 'wait for signal' action instead, where it waits for RF Channel to indicate it has disabled the item(s).
			//
			QThread::msleep(1000);

			/* Select and Set up the port that will be used */
			port = config->get_RCM_port_USB();
			Setup(port);

//			emit signal_RCM_notification("Configuring port: " + port);

			/* Connect mode-relevant communication signals and slots. */

			connect(RCM_port, &QSerialPort::readyRead, this, &RCM_Class::RCM_Live_ReadyRead_Handler);				//Live handler

			for (int i = 0; i < RF_Channels.count(); i++)
			{
				connect(this, &RCM_Class::signal_RCM_message, RCM_Class::RF_Channels.at(i), &RF_Channel::RCM_Live_serial_writeRead_to_SGx);
				connect(this, &RCM_Class::signal_RCM_sweep_message, RCM_Class::RF_Channels.at(i), &RF_Channel::RCM_Live_serial_Sweep_handler);
				connect(RF_Channels.at(i), &RF_Channel::signal_RCM_serial_response, this, &RCM_Class::RCM_Live_Response_from_SGX_Handler);
			}

			/* Open RCM port. */
			if (RCM_Open() == false)
			{
				Handler_RCM_Error();
				return;
			}

			emit signal_RCM_notification("USB Live mode enabled. Device is listening for user inputs at " + port);
			break;


		case RCM_MODE::RCM_MODE_USB_BLIND:

			/* Disable automatic error clearing and turn off the Readings Timer in RF Channel */
			for (int i = 1; i <= RF_Channels.count(); i++)
			{
				emit signal_set_error_clearing_enable(i, false);
				emit signal_timerStop(i);
			}

			/* Wait for above signals to take effect */
			//
			// TODO:
			// This msleep action is inelegant and not 100% reliable.
			// There should be a 'wait for signal' action instead, where it waits for RF Channel to indicate it has disabled the item(s).
			//
			QThread::msleep(1000);

			/* Select and Set up the port that will be used */
			port = config->get_RCM_port_USB();
			Setup(port);

			/* Connect mode-relevant communication signals and slots. */
			connect(RCM_port, &QSerialPort::readyRead, this, &RCM_Class::RCM_Blind_ReadyRead_Handler);		//Blind Handler

			for (int i = 0; i < RF_Channels.count(); i++)
			{
				qDebug() << "RCM: Disconnecting RF Channel[" << i+1 << "]'s Serial_Error_Handler.";
				disconnect(RCM_Class::RF_Channels.at(i)->SGX_port->serial, &QSerialPort::errorOccurred, RCM_Class::RF_Channels.at(i), &RF_Channel::Serial_Error_Handler);

				connect(this, &RCM_Class::signal_RCM_message, RCM_Class::RF_Channels.at(i), &RF_Channel::RCM_Blind_serial_write_to_SGx);
				connect(RCM_Class::RF_Channels.at(i)->SGX_port->serial, &QSerialPort::readyRead, RCM_Class::RF_Channels.at(i), &RF_Channel::RCM_Blind_serial_handleResponse_from_SGx);
				connect(RCM_Class::RF_Channels.at(i), &RF_Channel::signal_RCM_serial_response, this, &RCM_Class::RCM_Blind_Response_from_SGX_Handler);
			}

			/* Open RCM port. */
			if (RCM_Open() == false)
			{
				Handler_RCM_Error();
				return;
			}

			emit signal_RCM_notification("USB Blind mode enabled. Device is listening for user inputs at " + port);
			break;


		case RCM_MODE::RCM_MODE_TCP_LIVE:

			/* Disable Auto-Error Clearing */
			for (int i = 1; i <= RF_Channels.count(); i++)
			{
				emit signal_set_error_clearing_enable(i, false);
				emit signal_timerStart(i, config->get_polling_rate_data());
			}

			//
			// TODO:
			// This msleep action is inelegant and not 100% reliable.
			// There should be a 'wait for signal' action instead, where it waits for RF Channel to indicate it has disabled the item(s).
			//
			/* Wait for above signals to take effect */
			QThread::msleep(1000);

			/* Begin Listening for TCP Connections and print the viable IPs to connect to. */
			if (!RCM_TCP_Start_Listening())
			{
				Handler_RCM_TCP_Server_Error();
				return;
			}

			/* Unlike Serial connections that connect to an available port, there is no immediate connection available here.
			 * tcpServer must start listening for connections and then a user must connect (asynchronously) to tcpServer, and then things will get picked up from there. */

			break;



		case RCM_MODE::RCM_MODE_TCP_BLIND:

			/* Disable automatic error clearing and turn off the Readings Timer in RF Channel */
			for (int i = 1; i <= RF_Channels.count(); i++)
			{
				emit signal_set_error_clearing_enable(i, false);
				emit signal_timerStop(i);
			}

			//
			// TODO:
			// This msleep action is inelegant and not 100% reliable.
			// There should be a 'wait for signal' action instead, where it waits for RF Channel to indicate it has disabled the item(s).
			//
			/* Wait for above signals to take effect */
			QThread::msleep(1000);

			/* Begin Listening for TCP Connections and print the viable IPs to connect to. */
			if (!RCM_TCP_Start_Listening())
			{
				Handler_RCM_TCP_Server_Error();
				return;
			}

			/* Unlike Serial connections that connect to an available port, there is no immediate connection available here.
			 * tcpServer must start listening for connections and then a user must connect (asynchronously) to tcpServer, and then things will get picked up from there. */

			break;


		case RCM_MODE::RCM_MODE_RS232_LIVE:

			/* Disable Auto-Error Clearing */
			for (int i = 1; i <= RF_Channels.count(); i++)
			{
				emit signal_set_error_clearing_enable(i, false);
				emit signal_timerStart(i, config->get_polling_rate_data());
			}

			/* Wait for above signals to take effect */
			//
			// TODO:
			// This msleep action is inelegant and not 100% reliable.
			// There should be a 'wait for signal' action instead, where it waits for RF Channel to indicate it has disabled the item(s).
			//
			QThread::msleep(1000);

			/* Select and Set up the port that will be used */
			port = config->get_RCM_port_RS232();
			Setup(port);

			/* Connect mode-relevant communication signals and slots. */

			connect(RCM_port, &QSerialPort::readyRead, this, &RCM_Class::RCM_Live_ReadyRead_Handler);				//Live handler

			for (int i = 0; i < RF_Channels.count(); i++)
			{
				connect(this, &RCM_Class::signal_RCM_message, RCM_Class::RF_Channels.at(i), &RF_Channel::RCM_Live_serial_writeRead_to_SGx);
				connect(this, &RCM_Class::signal_RCM_sweep_message, RCM_Class::RF_Channels.at(i), &RF_Channel::RCM_Live_serial_Sweep_handler);
				connect(RF_Channels.at(i), &RF_Channel::signal_RCM_serial_response, this, &RCM_Class::RCM_Live_Response_from_SGX_Handler);

#if defined (Q_OS_LINUX)
				if (config->get_support_HAT_B14_0835())
				{
					GPIO_HAT_serial_mode->setValue("0");		//Sets RS232 mode
				}
#endif
			}

			/* Open RCM port. */
			if (RCM_Open() == false)
			{
				Handler_RCM_Error();
				return;
			}

			emit signal_RCM_notification("RS232 Live mode enabled. Device is listening for user inputs at " + port);
			break;


		case RCM_MODE::RCM_MODE_RS232_BLIND:

			/* Disable automatic error clearing and turn off the Readings Timer in RF Channel */
			for (int i = 1; i <= RF_Channels.count(); i++)
			{
				emit signal_set_error_clearing_enable(i, false);
				emit signal_timerStop(i);
			}

			/* Wait for above signals to take effect */
			//
			// TODO:
			// This msleep action is inelegant and not 100% reliable.
			// There should be a 'wait for signal' action instead, where it waits for RF Channel to indicate it has disabled the item(s).
			//
			QThread::msleep(1000);

			/* Select and Set up the port that will be used */
			port = config->get_RCM_port_RS232();
			Setup(port);

			/* Connect mode-relevant communication signals and slots. */
			connect(RCM_port, &QSerialPort::readyRead, this, &RCM_Class::RCM_Blind_ReadyRead_Handler);		//Blind Handler

			for (int i = 0; i < RF_Channels.count(); i++)
			{
				qDebug() << "RCM: Disconnecting RF Channel[" << i+1 << "]'s Serial_Error_Handler.";
				disconnect(RCM_Class::RF_Channels.at(i)->SGX_port->serial, &QSerialPort::errorOccurred, RCM_Class::RF_Channels.at(i), &RF_Channel::Serial_Error_Handler);

				connect(this, &RCM_Class::signal_RCM_message, RCM_Class::RF_Channels.at(i), &RF_Channel::RCM_Blind_serial_write_to_SGx);
				connect(RCM_Class::RF_Channels.at(i)->SGX_port->serial, &QSerialPort::readyRead, RCM_Class::RF_Channels.at(i), &RF_Channel::RCM_Blind_serial_handleResponse_from_SGx);
				connect(RCM_Class::RF_Channels.at(i), &RF_Channel::signal_RCM_serial_response, this, &RCM_Class::RCM_Blind_Response_from_SGX_Handler);
			}

#if defined (Q_OS_LINUX)
			if (config->get_support_HAT_B14_0835())
			{
				GPIO_HAT_serial_mode->setValue("0");		//Sets RS232 mode
			}
#endif

			/* Open RCM port. */
			if (RCM_Open() == false)
			{
				Handler_RCM_Error();
				return;
			}

			emit signal_RCM_notification("RS232 Blind mode enabled. Device is listening for user inputs at " + port);

			break;


		default:
			break;
	}

	/* Signal the engagement of selected Remote Command Mode */
	emit signal_RCM_mode(mode);

	/* If Applicable reset the SGx board.
	 * Avoid unnecessary resetting: like if the RCM port failed to initialize in the first place, there's no need to reset. */
	if (mode == RCM_MODE::RCM_MODE_OFF)
	{
		if (RCM_port->error() == QSerialPort::SerialPortError::NoError)
		{
#warning bug: if we switch from one interface to another but fail to initialize the new interface, it just goes to OFF but no reset occurs.
#warning The only case in which RCM should not auto-reset is if the initialization failed. RCM failures following any succesful communication probably do warrant a reset.

			if (config->get_reset_on_startup() == true)
			{
#warning multichannel systems are going to have a bad time! -> Should have a bit of wait time between resets.
				for (int i = 1; i <= RF_Channels.count(); i++)
				{
					emit signal_RCM_notification("Resetting system...");
					emit signal_execute_reset_SGx(i);
				}
			}
//			else
//			{
//				emit signal_RCM_notification("Auto-resetting is disabled. System state remains unchanged.");
//			}
		}
	}

	last_RCM_mode = mode;
}


/**********************************************************************************************************************************************************************************
* Remote Command Mode - TCP
* Supporting functions for Remote Command Mode over TCP.
 *********************************************************************************************************************************************************************************/
/* Make tcpServer start listening for connections and print the information necessary to connect to it. */
bool RCM_Class::RCM_TCP_Start_Listening()
{
	bool success = false;
	ipAddress = "";

	if(tcpServer->listen(QHostAddress::Any, config->get_RCM_port_TCP()))
	{
		success = true;

		/* Print IP information */
		QList<QHostAddress> ipAddressesList = QNetworkInterface::allAddresses();
		for (int i = 0; i < ipAddressesList.size(); ++i)
		{
			/* use the first non-localhost IPv4 address */
			if (ipAddressesList.at(i).toIPv4Address())
			{
				ipAddress += "-\t" + ipAddressesList.at(i).toString() + " : " + QString::number(tcpServer->serverPort());
				ipAddress += "\n\t";
			}
		}
		ipAddress.chop(2);		//Remove the excess characters at the end.

		emit signal_RCM_notification(QString("TCP Server is listening for connections at:\n\t%1").arg(ipAddress));
	}

	return success;
}

/* Handler for when a TCP client disconnects */
void RCM_Class::RCM_TCP_Client_Disconnected()
{
	QString peer_address = tcpSocket->peerAddress().toString();
	QStringList peer_address_trim = peer_address.split(":");
	qDebug() << "RCM: Client [" << peer_address_trim.last() << "] disconnected from the server.";

	/* Disconnect the signals and slots to avoid duplicate responses when next connection comes along. */
	tcpSocket->disconnectFromHost();
	Disconnect_Signals_Slots();

	if(tcpServer->isListening())
	{
		if (tcpServer->hasPendingConnections() == false)
		{
			emit signal_RCM_notification(QString("Client [%1] has disconnected.").arg(peer_address_trim.last()));
			emit signal_RCM_notification(QString("TCP Server is listening for connections at:\n\t%1").arg(ipAddress));
		}
	}
}

/* If a tcpSocket is already connected, clean it up */
void RCM_Class::RCM_TCP_Kick_Client()
{
	if(tcpSocket->state() == QTcpSocket::ConnectedState)
	{
		QString peer_address = tcpSocket->peerAddress().toString();
		QStringList peer_address_trim = peer_address.split(":");

		/* As soon as disconnectFromHost is called, the disconnected signal is emitted and RCM_TCP_Client_Disconnected() is activated immediately (before even finishing this function). */
		tcpSocket->disconnectFromHost();

		emit signal_RCM_notification(QString("Client [%1] has been disconnected.").arg(peer_address_trim.last()));
	}
}

/* Handler for new TCP connections to the TCP Server */
void RCM_Class::RCM_TCP_newConnection_Handler()
{
	/* If a tcpSocket is already connected, clean it up */
	RCM_TCP_Kick_Client();

	/* Connect the new tcpSocket */
	tcpSocket = tcpServer->nextPendingConnection();
	if(tcpSocket->state() == QTcpSocket::ConnectedState)
	{
		QString peer_address = tcpSocket->peerAddress().toString();
		QStringList peer_address_trim = peer_address.split(":");
		qDebug() << "RCM: New connection established with [" << peer_address_trim.last() << "]";
		emit signal_RCM_notification(QString("New connection established with client [%1].").arg(peer_address_trim.last()));
	}

	connect(tcpSocket, &QTcpSocket::disconnected, this, &RCM_Class::RCM_TCP_Client_Disconnected);


	if (last_RCM_mode == RCM_MODE::RCM_MODE_TCP_BLIND)
	{
		/* Connect mode-relevant communication signals and slots. */
		connect(tcpSocket, &QTcpSocket::readyRead, this, &RCM_Class::RCM_Blind_ReadyRead_Handler);		//Blind Handler

		for (int i = 0; i < RF_Channels.count(); i++)
		{
			qDebug() << "RCM: Disconnecting RF Channel[" << i+1 << "]'s Serial_Error_Handler.";
			disconnect(RCM_Class::RF_Channels.at(i)->SGX_port->serial, &QSerialPort::errorOccurred, RCM_Class::RF_Channels.at(i), &RF_Channel::Serial_Error_Handler);

			connect(this, &RCM_Class::signal_RCM_message, RCM_Class::RF_Channels.at(i), &RF_Channel::RCM_Blind_serial_write_to_SGx);
			connect(RCM_Class::RF_Channels.at(i)->SGX_port->serial, &QSerialPort::readyRead, RCM_Class::RF_Channels.at(i), &RF_Channel::RCM_Blind_serial_handleResponse_from_SGx);
			connect(RCM_Class::RF_Channels.at(i), &RF_Channel::signal_RCM_serial_response, this, &RCM_Class::RCM_Blind_Response_from_SGX_Handler);
		}
	}
	else if (last_RCM_mode == RCM_MODE::RCM_MODE_TCP_LIVE)
	{
		/* Connect mode-relevant communication signals and slots. */
		connect(tcpSocket, &QTcpSocket::readyRead, this, &RCM_Class::RCM_Live_ReadyRead_Handler);		//Blind Handler

		for (int i = 0; i < RF_Channels.count(); i++)
		{
			connect(this, &RCM_Class::signal_RCM_message, RCM_Class::RF_Channels.at(i), &RF_Channel::RCM_Live_serial_writeRead_to_SGx);
			connect(this, &RCM_Class::signal_RCM_sweep_message, RCM_Class::RF_Channels.at(i), &RF_Channel::RCM_Live_serial_Sweep_handler);
			connect(RCM_Class::RF_Channels.at(i), &RF_Channel::signal_RCM_serial_response, this, &RCM_Class::RCM_Live_Response_from_SGX_Handler);
		}
	}
}


/**********************************************************************************************************************************************************************************
* Remote Command Mode - BLIND
* This form of Remote Command Mode can be used when nothing else is using the serial line. The upside is that supports all communication.
* *******************************************************************************************************************************************************************************/
/* Handler for RCM Blind readyRead signal. Read a port's data and pass it to the handler in RF_Channel, which in turn sends it to the SGX_port. */
void RCM_Class::RCM_Blind_ReadyRead_Handler()
{
	QString message = RCM_Read_Blind();
	emit signal_RCM_message(message);
}

/* Handler for message responses from the SGx board */
void RCM_Class::RCM_Blind_Response_from_SGX_Handler(QString response)
{
	RCM_Write_Blind(response);
}

/* Read from the remote command interface's buffer. */
QString RCM_Class::RCM_Read_Blind()
{
	QString rx = "";

	switch(last_RCM_mode)
	{
		case RCM_MODE::RCM_MODE_TCP_BLIND:
			if (tcpSocket->isOpen())
			{
				rx += tcpSocket->readAll();
			}
			break;

		default:
			if (RCM_port->isOpen())
			{
				rx += QString::fromUtf8(RCM_port->readAll());
			}
			break;
	}

	qDebug() << "RCM RX:\t" << rx << "\n";
	return rx;
}

/* Write to the remote command interface's buffer */
void RCM_Class::RCM_Write_Blind(QString tx)
{
	switch(last_RCM_mode)
	{
		case RCM_MODE::RCM_MODE_TCP_BLIND:
			if(tcpSocket->isOpen())
			{
				tcpSocket->write((const char*)(tx).toUtf8());
			}
			break;

		default:
			if(RCM_port->isOpen())
			{
				RCM_port->write((const char*)(tx).toUtf8());
			}
			break;
	}

	qDebug() << "RCM TX:\t" << (tx) << "\n";
}

/**********************************************************************************************************************************************************************************
* Remote Command Mode - LIVE
* This form of remote command mode can be used alongside continous readings performed by RF Channel, but has limitations such as only supporting single-line responses.
 *********************************************************************************************************************************************************************************/
/*
 * Live RCM:
 * The RCM port raises its readyRead() signal -> A message has arrived at the port from the user -> Handle the incoming message.
 * 1. Read the message on the RCM port.
 * 2. Check the validity of the messsage's format; sending invalid messages to the SGx that don't yield a response will cause things to bug out.
 * 3. Check if SWP/$SWPD -> perform a workaround
 * 4. Pass the message on to the RF Channel(s) -> And onto the SGx board.
 * 5. (In RF Channel) Read the response from the SGx board and pass back to RCM_Live_Response_from_SGX_Handler
 * 6. Pass the response back to the RCM port.
 *
 * Remark: One RCM port may be used for multiple RF Channels.
 */

/* Live Handler for messages that appear on the RCM port from the user side */
void RCM_Class::RCM_Live_ReadyRead_Handler()
{
	static QString RCM_message = "";

	RCM_message += RCM_Read_Live();
	if ((RCM_message.contains("\r\n") || RCM_message.contains("\r") || RCM_message.contains("\n")) && RCM_message.contains("$"))
	{
		qDebug() << "RCM TX:\t" << RCM_message;
	}
	else
	{
		if (RCM_message == "\r\n" || RCM_message == "\r" || RCM_message == "\n")
		{
			RCM_message = ""; //Message was just a blank space, nothing more; clear the variable
		}
		return;
	}

	/* Make sure only one command is handled at a time */
	if (RCM_message.count("$") > 1)
	{
		RCM_Write_Live("RCM: One command at a time please.\r\n");
		RCM_message = ""; //Message was intercepted; clear the variable
		return;
	}

	if (!RCM_message.contains(QRegExp("\\$\\w{1,}\r*\n*\r*")))
	{
		qDebug() << "RCM: Command not valid.";
		RCM_Write_Live("RCM: Command not valid.\r\n");
		RCM_message = ""; //Message was intercepted; clear the variable
		return;
	}

	/* Disallow commands with multi-line returns and other stuff that doesn't conform to the protocol properly. */
	QStringList RCM_blacklist
	{
		"HELP", "HELPDEV",
		"CHANS", "CHANG",
		"IDLES",
		"EECSP", "EECSA"
	};

	for (int i = 0; i < RCM_blacklist.count(); i++)
	{
		if (RCM_message.contains(("$"+RCM_blacklist.at(i)), Qt::CaseInsensitive))
		{
			qDebug() << "RCM: This command is currently not supported by Live RCM";
			RCM_Write_Live("RCM: This command is currently not supported by Live RCM\r\n");
			RCM_message = ""; //Message was intercepted; clear the variable
			return;
		}
	}

	/* Special workaround for S11 sweeping, because it's just too important a feature to not support over RCM */
	if (RCM_message.contains("$SWP") || RCM_message.contains("$SWPD"))
	{
		QStringList received_message_list = RCM_message.split(",");

		/* Intercept sweep request with output mode 0 and use a special function; SWP has multi-line response which is currently not supported using the regular RCM method */
		// example of message: $SWP,0,2400,2500,10,50,0\r\n
		if (received_message_list.count() == 7)
		{
			if (received_message_list.last().toInt() == 0)
			{
				emit signal_RCM_sweep_message(RCM_message);
				RCM_message = ""; //Message was intercepted; clear the variable
				return;
			}
		}
	}

	emit signal_RCM_message(RCM_message);
	RCM_message = ""; //Message was sent; clear the variable
}


/* Live Handler for message responses from the SGx board */
void RCM_Class::RCM_Live_Response_from_SGX_Handler(QString response)
{
	RCM_Write_Live(response);
}


/* Write a message to the serial buffer */
void RCM_Class::RCM_Write_Live(QString tx)
{
	switch(last_RCM_mode)
	{
		case RCM_MODE::RCM_MODE_TCP_LIVE:
			if(tcpSocket->isOpen())
			{
				tcpSocket->write((const char*)(tx).toUtf8());
				tcpSocket->waitForBytesWritten(250);
			}
			break;

		default:
			if(RCM_port->isOpen())
			{
				RCM_port->write((const char*)(tx).toUtf8());
				RCM_port->waitForBytesWritten(250);
			}
	}

//	qDebug() << "TX:\t" << (const char*)(tx).toUtf8();
}

/* Read from the serial buffer */
QString RCM_Class::RCM_Read_Live()
{
	QString rx = "";

	switch(last_RCM_mode)
	{
		case RCM_MODE::RCM_MODE_TCP_LIVE:
			if (tcpSocket->isOpen())
			{
				rx += tcpSocket->readAll();
			}
			break;

		default:
			if (RCM_port->isOpen())
			{
				rx += QString::fromUtf8(RCM_port->readAll());
			}
	}

	/* Theoretically it is possible to run into the issue of rx filling up beyond the /r/n point, causing messages to be broken up.
	 * TX:     "$PPDG,1"
	 * RX:     "$PPDG,1,19.12035,20.93336\r\n$PP"
	 * TX:     "$PPDG,1"
	 * RX:     "DG,1,19.12035,21.00475\r\n$PPDG,1,19.09899,20.96905\r\n"
	 * this would pass the check, but results in a broken message. worst case multiple messages could get broken up in a row. */

	if(rx.count("\r\n") > 1)
	{
		qDebug() << "RCM: Buffer fill-up detected.";
	}

	qDebug() << "RX:\t" << rx << "\n";
	return rx;
}
